// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/*
 * This file contains information about particular algorithm suites used
 * within the encryption SDK.  In most cases, end-users don't need to
 * manipulate this structure, but it can occasionally be needed for more
 * advanced use cases, such as writing keyrings.
 *
 * Here we describe the overall shape of the Algorithm Suites used by the AWS Encryption
 * SDK for JavaScript.  Specific details for Node.js and WebCrypto can be found
 * in the respective files
 */
import { immutableClass, readOnlyProperty } from './immutable_class';
import { needs } from './needs';
/* References to https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/algorithms-reference.html
 * These define the possible parameters for algorithm specifications that correspond
 * to the Node.js or WebCrypto environment.
 * These parameters are composed into an algorithm suite specification for each
 * environment in the respective files.
 */
export var AlgorithmSuiteIdentifier;
(function (AlgorithmSuiteIdentifier) {
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES128_GCM_IV12_TAG16"] = 20] = "ALG_AES128_GCM_IV12_TAG16";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES192_GCM_IV12_TAG16"] = 70] = "ALG_AES192_GCM_IV12_TAG16";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16"] = 120] = "ALG_AES256_GCM_IV12_TAG16";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256"] = 276] = "ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256"] = 326] = "ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256"] = 376] = "ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256"] = 532] = "ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384"] = 838] = "ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384"] = 888] = "ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY"] = 1144] = "ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384"] = 1400] = "ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384";
})(AlgorithmSuiteIdentifier || (AlgorithmSuiteIdentifier = {}));
Object.freeze(AlgorithmSuiteIdentifier);
export var CommitmentPolicy;
(function (CommitmentPolicy) {
    CommitmentPolicy["FORBID_ENCRYPT_ALLOW_DECRYPT"] = "FORBID_ENCRYPT_ALLOW_DECRYPT";
    CommitmentPolicy["REQUIRE_ENCRYPT_ALLOW_DECRYPT"] = "REQUIRE_ENCRYPT_ALLOW_DECRYPT";
    CommitmentPolicy["REQUIRE_ENCRYPT_REQUIRE_DECRYPT"] = "REQUIRE_ENCRYPT_REQUIRE_DECRYPT";
})(CommitmentPolicy || (CommitmentPolicy = {}));
Object.freeze(CommitmentPolicy);
export var SignaturePolicy;
(function (SignaturePolicy) {
    SignaturePolicy["ALLOW_ENCRYPT_ALLOW_DECRYPT"] = "ALLOW_ENCRYPT_ALLOW_DECRYPT";
    SignaturePolicy["ALLOW_ENCRYPT_FORBID_DECRYPT"] = "ALLOW_ENCRYPT_FORBID_DECRYPT";
})(SignaturePolicy || (SignaturePolicy = {}));
Object.freeze(SignaturePolicy);
export const NonSigningAlgorithmSuiteIdentifier = (() => {
    const { ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256, ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384, 
    // Both the name side above, and the id side below
    [0x0214]: NAME_ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256, [0x0346]: NAME_ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, [0x0378]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, [0x0578]: NAME_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384, ...NonSigningAlgorithmSuiteIdentifier } = AlgorithmSuiteIdentifier;
    return NonSigningAlgorithmSuiteIdentifier;
})();
export const SignaturePolicySuites = Object.freeze({
    isDecryptEnabled(signaturePolicy, suite, messageId) {
        const id = suite.id || suite;
        const name = suite.name || AlgorithmSuiteIdentifier[id];
        let decryption_client_name = 'decryptStream';
        let signature_description = 'signed';
        if (signaturePolicy === SignaturePolicy.ALLOW_ENCRYPT_FORBID_DECRYPT) {
            decryption_client_name = 'decryptUnsignedMessageStream';
            signature_description = 'un-signed';
        }
        /* Precondition: Only handle DecryptionMaterial for algorithm suites supported in signaturePolicy. */
        needs(this[signaturePolicy].decryptEnabledSuites[id], `Configuration conflict. ` +
            `Cannot process message with ID ${messageId} ` +
            `due to client method ${decryption_client_name} ` +
            `requiring only ${signature_description} messages. ` +
            `Algorithm ID was ${name}. ` +
            `See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/concepts.html#digital-sigs`);
    },
    [SignaturePolicy.ALLOW_ENCRYPT_ALLOW_DECRYPT]: Object.freeze({
        decryptEnabledSuites: AlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    }),
    [SignaturePolicy.ALLOW_ENCRYPT_FORBID_DECRYPT]: Object.freeze({
        decryptEnabledSuites: NonSigningAlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY,
    }),
});
export const NonCommittingAlgorithmSuiteIdentifier = (() => {
    const { ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384, 
    // Both the name side above, and the id side below
    [0x0478]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY, [0x0578]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384, ...NonCommittingAlgorithmSuiteIdentifier } = AlgorithmSuiteIdentifier;
    return NonCommittingAlgorithmSuiteIdentifier;
})();
export const CommittingAlgorithmSuiteIdentifier = (() => {
    const { ALG_AES128_GCM_IV12_TAG16, ALG_AES192_GCM_IV12_TAG16, ALG_AES256_GCM_IV12_TAG16, ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256, ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256, ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256, ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, 
    // Both the name side above, and the id side below
    [0x0014]: NAME_ALG_AES128_GCM_IV12_TAG16, [0x0046]: NAME_ALG_AES192_GCM_IV12_TAG16, [0x0078]: NAME_ALG_AES256_GCM_IV12_TAG16, [0x0114]: NAME_ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256, [0x0146]: NAME_ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256, [0x0178]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256, [0x0214]: NAME_ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256, [0x0346]: NAME_ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, [0x0378]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, ...CommittingAlgorithmSuiteIdentifier } = AlgorithmSuiteIdentifier;
    return CommittingAlgorithmSuiteIdentifier;
})();
export const CommitmentPolicySuites = Object.freeze({
    isEncryptEnabled(commitmentPolicy, suite) {
        if (!suite)
            return;
        const id = suite.id || suite;
        const name = suite.name || AlgorithmSuiteIdentifier[id];
        /* Precondition: Only handle EncryptionMaterial for algorithm suites supported in commitmentPolicy. */
        needs(this[commitmentPolicy].encryptEnabledSuites[id], `Configuration conflict. ` +
            `Cannot encrypt due to CommitmentPolicy ${commitmentPolicy} ` +
            `requiring only non-committed messages. ` +
            `Algorithm ID was ${name}. ` +
            `See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/troubleshooting-migration.html`);
    },
    isDecryptEnabled(commitmentPolicy, suite, messageId) {
        const id = suite.id || suite;
        const name = suite.name || AlgorithmSuiteIdentifier[id];
        /* Precondition: Only handle DecryptionMaterial for algorithm suites supported in commitmentPolicy. */
        needs(this[commitmentPolicy].decryptEnabledSuites[id], `Configuration conflict. ` +
            `Cannot process message with ID ${messageId} ` +
            `due to CommitmentPolicy ${commitmentPolicy} ` +
            `requiring only committed messages. ` +
            `Algorithm ID was ${name}. ` +
            `See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/troubleshooting-migration.html`);
    },
    [CommitmentPolicy.FORBID_ENCRYPT_ALLOW_DECRYPT]: Object.freeze({
        encryptEnabledSuites: NonCommittingAlgorithmSuiteIdentifier,
        decryptEnabledSuites: AlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: NonCommittingAlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    }),
    [CommitmentPolicy.REQUIRE_ENCRYPT_ALLOW_DECRYPT]: Object.freeze({
        encryptEnabledSuites: CommittingAlgorithmSuiteIdentifier,
        decryptEnabledSuites: AlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: CommittingAlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    }),
    [CommitmentPolicy.REQUIRE_ENCRYPT_REQUIRE_DECRYPT]: Object.freeze({
        encryptEnabledSuites: CommittingAlgorithmSuiteIdentifier,
        decryptEnabledSuites: CommittingAlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: CommittingAlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    }),
});
export var MessageFormat;
(function (MessageFormat) {
    MessageFormat[MessageFormat["V1"] = 1] = "V1";
    MessageFormat[MessageFormat["V2"] = 2] = "V2";
})(MessageFormat || (MessageFormat = {}));
Object.freeze(MessageFormat);
export class AlgorithmSuite {
    constructor(suiteValues) {
        needs(this.constructor !== AlgorithmSuite, 'new AlgorithmSuite is not allowed');
        /* Precondition: A algorithm suite specification must be passed. */
        needs(suiteValues, 'Algorithm specification not set.');
        /* Precondition: The Algorithm Suite Identifier must exist. */
        needs(AlgorithmSuiteIdentifier[suiteValues.id], 'No suite by that identifier exists.');
        Object.assign(this, suiteValues);
        readOnlyProperty(this, 'keyLengthBytes', this.keyLength / 8);
        readOnlyProperty(this, 'name', AlgorithmSuiteIdentifier[this.id]);
    }
}
immutableClass(AlgorithmSuite);
//# sourceMappingURL=data:application/json;base64,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