/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { CodebaseInvestigatorAgent } from './codebase-investigator.js';
import {} from 'zod';
/**
 * Manages the discovery, loading, validation, and registration of
 * AgentDefinitions.
 */
export class AgentRegistry {
    config;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    agents = new Map();
    constructor(config) {
        this.config = config;
    }
    /**
     * Discovers and loads agents.
     */
    async initialize() {
        this.loadBuiltInAgents();
        if (this.config.getDebugMode()) {
            console.log(`[AgentRegistry] Initialized with ${this.agents.size} agents.`);
        }
    }
    loadBuiltInAgents() {
        this.registerAgent(CodebaseInvestigatorAgent);
    }
    /**
     * Registers an agent definition. If an agent with the same name exists,
     * it will be overwritten, respecting the precedence established by the
     * initialization order.
     */
    registerAgent(definition) {
        // Basic validation
        if (!definition.name || !definition.description) {
            console.warn(`[AgentRegistry] Skipping invalid agent definition. Missing name or description.`);
            return;
        }
        if (this.agents.has(definition.name) && this.config.getDebugMode()) {
            console.log(`[AgentRegistry] Overriding agent '${definition.name}'`);
        }
        this.agents.set(definition.name, definition);
    }
    /**
     * Retrieves an agent definition by name.
     */
    getDefinition(name) {
        return this.agents.get(name);
    }
    /**
     * Returns all active agent definitions.
     */
    getAllDefinitions() {
        return Array.from(this.agents.values());
    }
}
//# sourceMappingURL=registry.js.map