﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/athena/AthenaRequest.h>
#include <aws/athena/Athena_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Athena {
namespace Model {

/**
 */
class GetPreparedStatementRequest : public AthenaRequest {
 public:
  AWS_ATHENA_API GetPreparedStatementRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetPreparedStatement"; }

  AWS_ATHENA_API Aws::String SerializePayload() const override;

  AWS_ATHENA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the prepared statement to retrieve.</p>
   */
  inline const Aws::String& GetStatementName() const { return m_statementName; }
  inline bool StatementNameHasBeenSet() const { return m_statementNameHasBeenSet; }
  template <typename StatementNameT = Aws::String>
  void SetStatementName(StatementNameT&& value) {
    m_statementNameHasBeenSet = true;
    m_statementName = std::forward<StatementNameT>(value);
  }
  template <typename StatementNameT = Aws::String>
  GetPreparedStatementRequest& WithStatementName(StatementNameT&& value) {
    SetStatementName(std::forward<StatementNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The workgroup to which the statement to be retrieved belongs.</p>
   */
  inline const Aws::String& GetWorkGroup() const { return m_workGroup; }
  inline bool WorkGroupHasBeenSet() const { return m_workGroupHasBeenSet; }
  template <typename WorkGroupT = Aws::String>
  void SetWorkGroup(WorkGroupT&& value) {
    m_workGroupHasBeenSet = true;
    m_workGroup = std::forward<WorkGroupT>(value);
  }
  template <typename WorkGroupT = Aws::String>
  GetPreparedStatementRequest& WithWorkGroup(WorkGroupT&& value) {
    SetWorkGroup(std::forward<WorkGroupT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_statementName;

  Aws::String m_workGroup;
  bool m_statementNameHasBeenSet = false;
  bool m_workGroupHasBeenSet = false;
};

}  // namespace Model
}  // namespace Athena
}  // namespace Aws
