/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*-
 *
 * Copyright 2025 GNOME Foundation, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *  - Ignacy Kuchciński <ignacykuchcinski@gnome.org>
 */

#include "config.h"

#include <glib/gi18n-lib.h>
#include <libmalcontent-ui/malcontent-ui.h>

#include "access-page.h"

/**
 * MctAccessPage:
 *
 * A widget which shows parental controls access and restrictions
 * for the selected user.
 *
 * [property@Malcontent.AccessPage:user] may be `NULL`, in which case the
 * contents of the widget are undefined and it should not be shown.
 *
 * Since: 0.14.0
 */
struct _MctAccessPage
{
  AdwNavigationPage parent;

  AdwWindowTitle *access_window_title;
  MctUserControls *user_controls;

  MctUser *user; /* (owned) (nullable) */
  unsigned long user_notify_id;
};

G_DEFINE_TYPE (MctAccessPage, mct_access_page, ADW_TYPE_NAVIGATION_PAGE)

typedef enum
{
  PROP_USER = 1,
} MctAccessPageProperty;

static GParamSpec *properties[PROP_USER + 1];

static void
mct_access_page_get_property (GObject    *object,
                              guint       prop_id,
                              GValue     *value,
                              GParamSpec *pspec)
{
  MctAccessPage *self = MCT_ACCESS_PAGE (object);

  switch ((MctAccessPageProperty) prop_id)
    {
      case PROP_USER:
        g_value_set_object (value, self->user);
        break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
mct_access_page_set_property (GObject      *object,
                              guint         prop_id,
                              const GValue *value,
                              GParamSpec   *pspec)
{
  MctAccessPage *self = MCT_ACCESS_PAGE (object);

  switch ((MctAccessPageProperty) prop_id)
    {
      case PROP_USER:
        mct_access_page_set_user (self, g_value_dup_object (value), NULL);
        break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
mct_access_page_dispose (GObject *object)
{
  MctAccessPage *self = MCT_ACCESS_PAGE (object);

  g_clear_signal_handler (&self->user_notify_id, self->user);
  g_clear_object (&self->user);

  G_OBJECT_CLASS (mct_access_page_parent_class)->dispose (object);
}

static void
mct_access_page_class_init (MctAccessPageClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

  object_class->get_property = mct_access_page_get_property;
  object_class->set_property = mct_access_page_set_property;
  object_class->dispose = mct_access_page_dispose;

  /**
   * MctAccessPage:user: (nullable)
   *
   * The currently selected user account.
   *
   * Since 0.14.0
   */
  properties[PROP_USER] =
      g_param_spec_object ("user", NULL, NULL,
                           MCT_TYPE_USER,
                           G_PARAM_READWRITE |
                           G_PARAM_STATIC_STRINGS |
                           G_PARAM_EXPLICIT_NOTIFY);

  g_object_class_install_properties (object_class, G_N_ELEMENTS (properties), properties);

  gtk_widget_class_set_template_from_resource (widget_class, "/org/freedesktop/MalcontentControl/ui/access-page.ui");

  gtk_widget_class_bind_template_child (widget_class, MctAccessPage, access_window_title);
  gtk_widget_class_bind_template_child (widget_class, MctAccessPage, user_controls);
}

static void
mct_access_page_init (MctAccessPage *self)
{
  gtk_widget_init_template (GTK_WIDGET (self));
}

/**
 * mct_access_page_new:
 *
 * Create a new [class@Malcontent.AccessPage] widget.
 *
 * Returns: (transfer full): a new access page
 * Since: 0.14.0
 */
MctAccessPage *
mct_access_page_new (void)
{
  return g_object_new (MCT_TYPE_ACCESS_PAGE,
                       NULL);
}

/**
 * mct_access_page_get_user:
 * @self: an access page
 *
 * Get the currently selected user.
 *
 * Returns: (transfer none) (nullable): the currently selected user
 * Since: 0.14.0
 */
MctUser *
mct_access_page_get_user (MctAccessPage *self)
{
  g_return_val_if_fail (MCT_IS_ACCESS_PAGE (self), NULL);

  return self->user;
}

static void
user_notify_cb (GObject    *object,
                GParamSpec *pspec,
                void       *user_data)
{
  MctUser *user = MCT_USER (object);
  MctAccessPage *self = MCT_ACCESS_PAGE (user_data);
  g_autofree gchar *help_label = NULL;

  adw_window_title_set_subtitle (self->access_window_title,
                                 mct_user_get_display_name (user));

  /* Translators: Replace the link to commonsensemedia.org with some
   * localised guidance for parents/carers on how to set restrictions on
   * their child/caree in a responsible way which is in keeping with the
   * best practice and culture of the region. If no suitable localised
   * guidance exists, and if the default commonsensemedia.org link is not
   * suitable, please file an issue against malcontent so we can discuss
   * further!
   * https://gitlab.freedesktop.org/pwithnall/malcontent/-/issues/new
   */
  help_label = g_strdup_printf (_("It’s recommended that restrictions are "
                                  "set as part of an ongoing conversation "
                                  "with %s. <a href='https://www.commonsensemedia.org/privacy-and-internet-safety'>"
                                  "Read guidance</a> on what to consider."),
                                mct_user_get_display_name (user));

  mct_user_controls_set_description (self->user_controls, help_label);
}

/**
 * mct_access_page_set_user:
 * @self: an access page
 * @user: (nullable): a user
 * @permission: (nullable) (transfer none): the [class@Gio.Permission]
 *   indicating whether the current user has permission to view or change
 *   parental controls, or `NULL` if permission is not allowed or is unknown
 *
 * Set the currently selected user.
 *
 * Since: 0.14.0
 */
void
mct_access_page_set_user (MctAccessPage *self,
                          MctUser       *user,
                          GPermission   *permission)
{
  g_return_if_fail (MCT_IS_ACCESS_PAGE (self));
  g_return_if_fail (user == NULL || MCT_IS_USER (user));
  g_return_if_fail (permission == NULL || G_IS_PERMISSION (permission));

  g_autoptr(MctUser) old_user = NULL;

  old_user = (self->user != NULL) ? g_object_ref (self->user) : NULL;

  if (g_set_object (&self->user, user))
    {
      if (old_user != NULL)
        g_clear_signal_handler (&self->user_notify_id, old_user);

      if (user != NULL)
        {
          self->user_notify_id = g_signal_connect (user,
                                                   "notify",
                                                   G_CALLBACK (user_notify_cb),
                                                   self);
          user_notify_cb (G_OBJECT (user), NULL, self);
        }

      mct_user_controls_set_permission (self->user_controls, permission);
      mct_user_controls_set_user (self->user_controls, user);

      g_object_notify_by_pspec (G_OBJECT (self), properties[PROP_USER]);
    }
}
