// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A1__F216

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_32_3_F_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1100)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xfe54fa4au, 0xd2ba3282u, 0, 46 }, // fe54fa4ad2ba3282 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1100
    { 0xbc7988b1u, 0x2634f2f3u, 40, 46 }, // bc7988b12634f2f3 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1100
    { 0xc5410d58u, 0x37e80ec9u, 40, 63 }, // c5410d5837e80ec9 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1100
    { 0x84fdfd63u, 0x83b90838u, 40, 6 }, // 84fdfd6383b90838 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1100
    { 0xf68fe8b1u, 0xc2420df3u, 40, 23 }, // f68fe8b1c2420df3 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1100
    { 0xf743e873u, 0xeb5d74adu, 97, 138 }, // f743e873eb5d74ad = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1100
    { 0x1f9b86d1u, 0x9906d156u, 97, 80 }, // 1f9b86d19906d156 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1100
    { 0x2b3209ddu, 0xc2bb335cu, 97, 155 }, // 2b3209ddc2bb335c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1100
    { 0x905497c0u, 0xb0100d98u, 97, 103 }, // 905497c0b0100d98 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1100
    { 0xc1c48b1au, 0x38229268u, 120, 80 }, // c1c48b1a38229268 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1100
    { 0xfb53d525u, 0x0c1f9565u, 120, 155 }, // fb53d5250c1f9565 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1100
    { 0x14bdb337u, 0xd8bb3072u, 120, 103 }, // 14bdb337d8bb3072 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1100
    { 0x62812738u, 0x0607c1a3u, 126, 80 }, // 628127380607c1a3 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1100
    { 0xea287667u, 0x4c3b6e9au, 126, 103 }, // ea2876674c3b6e9a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_32_3_F_F_0___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1100
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1100_mod0
{{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
 { 0, 6, 7, 5, 8, 0, 6, 0, 6, 5},
 { 4, 1, 0, 0, 0, 0, 0, 0, 0, 0},
 { 2, 2, 4, 4, 2, 2, 4, 3, 3, 4},
 { 3, 2, 4, 1, 4, 4, 2, 4, 3, 1},
 { 3, 1, 1, 3, 3, 2, 3, 3, 4, 1},
 { 1, 4, 4, 1, 4, 2,11,10, 9, 8},
 { 1, 4, 3, 2,11,11, 9,10,13,12},
 { 1, 2, 2,10, 9, 9,10, 9,10,11},
 { 4, 3, 2, 9,10,10, 2,11, 9,11}}
// End of GPU gfx1100_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A1__F216 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 4;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 4;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = "i32"
// BLOCK_DMODEL = 32
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 32 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

