/*****************************************************************************\
 *  sluid.h - Slurm Lexicographically-sortable Unique ID
 *****************************************************************************
 *  This file is part of Slurm, a resource management program.
 *  For details, see <https://slurm.schedmd.com/>.
 *  Please also read the included file: DISCLAIMER.
 *
 *  Slurm is free software; you can redistribute it and/or modify it under
 *  the terms of the GNU General Public License as published by the Free
 *  Software Foundation; either version 2 of the License, or (at your option)
 *  any later version.
 *
 *  In addition, as a special exception, the copyright holders give permission
 *  to link the code of portions of this program with the OpenSSL library under
 *  certain conditions as described in each individual source file, and
 *  distribute linked combinations including the two. You must obey the GNU
 *  General Public License in all respects for all of the code used other than
 *  OpenSSL. If you modify file(s) with this exception, you may extend this
 *  exception to your version of the file(s), but you are not obligated to do
 *  so. If you do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source files in
 *  the program, then also delete it here.
 *
 *  Slurm is distributed in the hope that it will be useful, but WITHOUT ANY
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 *  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with Slurm; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA.
\*****************************************************************************/

#ifndef _COMMON_SLUID_H
#define _COMMON_SLUID_H

#include "slurm/slurm.h"

/*
 * A SLUID is a 64-bit job identifier that is unique within a given Slurm realm.
 *
 * They are generated by the slurmctld, based on a system identifier that is
 * guaranteed to be unique by the slurmdbd.
 * They can be turned into UUIDs for interoperability with external services.
 *
 * The canonical printed form is "Crockford's Base32" ("cb32").
 * This allows for a standardized 14-character format that (a) sorts
 * lexicographically, (b) avoids issues with upper-vs-lower case, and (c) is
 * distinct from any other JobId format in use in Slurm due to the leading 's'.
 *
 * NOTES: All diagrams are big-endian.
 *
 * The SLUID is constructed as:
 * [ 12-bit cluster ][ 42-bit unix_ts_ms ][ 10-bit sequence ]
 *
 * The cluster field is established by slurmdbd for each slurmctld.
 * This does limit a slurmdbd installation to 4093 distinct clusters.
 * - 0x000 is reserved to avoid collisions with db_index values.
 * - 0x001 is reserved for slurmdbd. Also used for --no-allocate.
 * - 0xfff is reserved for future use.
 *
 * The timestamp is milliseconds since the unix epoch. It is stored in the job
 * state file to ensure it is always incrementing. Slurm's implementation will
 * last until Y2106, at which point someone will hopefully address the ensuing
 * overflow into the cluster bits.
 *
 * The sequence value is incremented if multiple SLUIDs are generated within
 * the same millisecond. This is uncommon for individually-submitted jobs, but
 * can be observed when submitting job arrays which pre-reserve their entire
 * range at job submission time.
 *
 * The design for SLUID was inspired by:
 * - https://github.com/flux-framework/rfc/blob/master/spec_19.rst
 * - https://github.com/ulid/spec
 * - https://www.crockford.com/base32.html
 * - https://www.rfc-editor.org/rfc/rfc9562.html
 */

/* Bytes required to hold a stringified sluid_t including NUL termination. */
#define SLUID_STR_BYTES 15

/*
 * Initialize generator.
 */
extern void sluid_init(uint16_t cluster, time_t minimum);

/*
 * Generate a random cluster id.
 */
extern uint16_t generate_cluster_id(void);

/*
 * Generate new SLUID.
 * Will fatal() if sluid_init() has never been called.
 * Will also fatal() if clock_gettime() fails.
 */
extern sluid_t generate_sluid(void);

/*
 * The canonical printed form for a SLUID is a lower-case "s" followed by
 * 13 characters encoding the numerical value in "Crockford's Base32".
 *
 * Returns an xmalloc()'d string.
 */
extern char *sluid2str(const sluid_t sluid);

/*
 * Directly print into an existing buffer.
 * Must have at least 15 bytes free or nothing will be printed.
 */
extern void print_sluid(const sluid_t sluid, char *buffer, size_t size);

/*
 * Parse the string representation.
 * Must be "s", followed by 13 digits encoding the SLUID, then NUL.
 * Returns 0 on error.
 */
extern sluid_t str2sluid(const char *string);

/*
 * Any SLUID can be promoted into a (mostly-compliant) UUIDv7 (rfc9562).
 *
 * UUIDv7 is:
 * [ 48-bit unix_ts_ms ][ 4-bit version (0b0111) ][ 12-bit rand_a ]
 * [ 2-bit var (0b10) ][ 62-bit rand_b ]
 *
 * SLUID-in-UUIDv7 is:
 * [ 48-bit unix_ts_ms ][ 4-bit version (0b0111) ][ 12 bit sequence ]
 * [ 2-bit var (0b10) ][ 12-bit cluster ][ 18-bit padding ][ 32-bit step ]
 *
 * The 18-bit padding is generated once per cluster, maintained in the state
 * file and accounting, and kept throughout the cluster lifespan. This
 * permits both slurmctld and slurmdbd to generate and validate these UUIDs.
 *
 * The step field is 0xffffffff for anything except job steps.
 *
 * Returns an xmalloc()'d string.
 */
extern char *sluid2uuid(const sluid_t sluid, const uint64_t padding);
extern char *stepid2uuid(const slurm_step_id_t step, const uint64_t padding);

#endif
