// ----------------------------------------------------------------------------
//
//  Copyright (C) 2024 Fons Adriaensen <fons@linuxaudio.org>
//    
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// ----------------------------------------------------------------------------


#include <Python.h>
#include "jfwplay.h"


static const char *capslabel = "Jfwplay";


extern "C" void destroy (PyObject *P)
{
    delete (Jfwplay *) PyCapsule_GetPointer (P, capslabel);
}


extern "C" PyObject* makecaps (PyObject *self, PyObject *args)
{
    Jfwplay   *J;
    PyObject  *P;
    const char *client_name;
    const char *server_name;
    int nchan;
    
    if (! PyArg_ParseTuple(args, "Oszi", &P, &client_name, &server_name,
			   &nchan)) return NULL;
    J = new Jfwplay (client_name, server_name, nchan);
    return Py_BuildValue ("NN",
			  PyCapsule_New (J, capslabel, destroy),
                          PyCapsule_New (dynamic_cast<Jclient *>(J), "Jclient", 0));
}


extern "C" PyObject* load_file (PyObject *self, PyObject *args)
{
    PyObject  *P;
    Jfwplay   *J;
    char      *filename;

    if (! PyArg_ParseTuple(args, "Oz", &P, &filename)) return 0;
    J = (Jfwplay *) PyCapsule_GetPointer (P, capslabel);
    return Py_BuildValue ("i", J->load_file (filename));
}


extern "C" PyObject* set_range (PyObject *self, PyObject *args)
{
    PyObject   *P;
    Jfwplay    *J;
    int64_t    n0, n1;
    
    if (! PyArg_ParseTuple (args, "OLL", &P, &n0, &n1)) return 0;
    J = (Jfwplay *) PyCapsule_GetPointer (P, capslabel);
    return Py_BuildValue ("i", J->set_range (n0, n1));
}


extern "C" PyObject* set_state (PyObject *self, PyObject *args)
{
    PyObject  *P;
    Jfwplay   *J;
    int       state;

    if (! PyArg_ParseTuple(args, "Oi", &P, &state)) return NULL;
    J = (Jfwplay *) PyCapsule_GetPointer (P, capslabel);
    return Py_BuildValue ("i", J->set_state (state));
}


extern "C" PyObject* fileinfo (PyObject *self, PyObject *args)
{
    PyObject  *P;
    Jfwplay   *J;

    if (! PyArg_ParseTuple(args, "O", &P)) return NULL;
    J = (Jfwplay *) PyCapsule_GetPointer (P, capslabel);
    return Py_BuildValue ("Li", J->fsize (), J->fchan ());
}


static PyMethodDef JackFwplayMethods[] =
{ 
    { "makecaps",        makecaps,        METH_VARARGS, "Create object capsules."    },
    { "load_file",       load_file,       METH_VARARGS, "Open file for playback."    },
    { "set_range",       set_range,       METH_VARARGS, "Set playback sample range." },
    { "set_state",       set_state,       METH_VARARGS, "Set process state."         },
    { "fileinfo",        fileinfo,        METH_VARARGS, "Return file shape."         },
    {NULL, NULL, 0, NULL}
};



#if PY_VERSION_HEX >= 0x03000000

static struct PyModuleDef JackFwplayModule = 
{
   PyModuleDef_HEAD_INIT,
   "jackfwplay_ext",
   NULL, 
   -1, 
   JackFwplayMethods
};

PyMODINIT_FUNC PyInit_jackfwplay_ext(void)
{
    return PyModule_Create (&JackFwplayModule);
}

#else
#error "This code requires Python 3."
#endif
