/*
-----------------------------------------------------------------------------
This source file is part of OGRE
(Object-oriented Graphics Rendering Engine)
For the latest info, see http://www.ogre3d.org/

Copyright (c) 2000-2014 Torus Knot Software Ltd

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
-----------------------------------------------------------------------------
*/

#ifndef __PrefabFactory_H__
#define __PrefabFactory_H__

#include "OgrePrerequisites.h"

namespace Ogre {

    /** \addtogroup Core
    *  @{
    */
    /** \addtogroup Resources
    *  @{
    */

    /** Enum identifying the types of manual mesh built by this manager */
    enum MeshBuildType
    {
        MBT_PLANE,
        MBT_CURVED_ILLUSION_PLANE,
        MBT_CURVED_PLANE
    };

    /** Saved parameters used to (re)build a manual mesh built by this class */
    struct MeshBuildParams
    {
        MeshBuildType type;
        Plane plane;
        Real width;
        Real height;
        Real curvature;
        int xsegments;
        int ysegments;
        bool normals;
        unsigned short numTexCoordSets;
        Real xTile;
        Real yTile;
        Vector3 upVector;
        Quaternion orientation;
        HardwareBuffer::Usage vertexBufferUsage;
        HardwareBuffer::Usage indexBufferUsage;
        bool vertexShadowBuffer;
        bool indexShadowBuffer;
        int ySegmentsToKeep;
    };

    /** A factory class that can create various mesh prefabs. 

        This class is used by OgreMeshManager to offload the loading of various prefab types 
        to a central location.
    */
    class PrefabFactory : public ManualResourceLoader
    {
    public:
        /** If the given mesh has a UserObjectBinding for _MeshBuildParams or resource name (e.g "Prefab_Plane")
            then this prefab will be created as a submesh of the given mesh.

            @param res The mesh that the potential prefab will be created in.
        */
        void loadResource(Resource* res) override;

    private:
        /// Creates a plane as a submesh of the given mesh
        static void createPlane(Mesh* mesh);

        /// Creates a 100x100x100 cube as a submesh of the given mesh
        static void createCube(Mesh* mesh);

        /// Creates a sphere with a diameter of 100 units as a submesh of the given mesh
        static void createSphere(Mesh* mesh);

        /** Utility method for manual loading a plane */
        static void loadManualPlane(Mesh* pMesh, MeshBuildParams& params);
        /** Utility method for manual loading a curved plane */
        static void loadManualCurvedPlane(Mesh* pMesh, MeshBuildParams& params);
        /** Utility method for manual loading a curved illusion plane */
        static void loadManualCurvedIllusionPlane(Mesh* pMesh, MeshBuildParams& params);
    };
    /** @} */
    /** @} */

} // namespace Ogre

#endif
