{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A render node drawing a set of glyphs.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gsk.Objects.TextNode
    ( 

-- * Exported types
    TextNode(..)                            ,
    IsTextNode                              ,
    toTextNode                              ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [draw]("GI.Gsk.Objects.RenderNode#g:method:draw"), [hasColorGlyphs]("GI.Gsk.Objects.TextNode#g:method:hasColorGlyphs"), [ref]("GI.Gsk.Objects.RenderNode#g:method:ref"), [serialize]("GI.Gsk.Objects.RenderNode#g:method:serialize"), [unref]("GI.Gsk.Objects.RenderNode#g:method:unref"), [writeToFile]("GI.Gsk.Objects.RenderNode#g:method:writeToFile").
-- 
-- ==== Getters
-- [getBounds]("GI.Gsk.Objects.RenderNode#g:method:getBounds"), [getColor]("GI.Gsk.Objects.TextNode#g:method:getColor"), [getFont]("GI.Gsk.Objects.TextNode#g:method:getFont"), [getGlyphs]("GI.Gsk.Objects.TextNode#g:method:getGlyphs"), [getNodeType]("GI.Gsk.Objects.RenderNode#g:method:getNodeType"), [getNumGlyphs]("GI.Gsk.Objects.TextNode#g:method:getNumGlyphs"), [getOffset]("GI.Gsk.Objects.TextNode#g:method:getOffset"), [getOpaqueRect]("GI.Gsk.Objects.RenderNode#g:method:getOpaqueRect").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveTextNodeMethod                   ,
#endif

-- ** getColor #method:getColor#

#if defined(ENABLE_OVERLOADING)
    TextNodeGetColorMethodInfo              ,
#endif
    textNodeGetColor                        ,


-- ** getFont #method:getFont#

#if defined(ENABLE_OVERLOADING)
    TextNodeGetFontMethodInfo               ,
#endif
    textNodeGetFont                         ,


-- ** getGlyphs #method:getGlyphs#

#if defined(ENABLE_OVERLOADING)
    TextNodeGetGlyphsMethodInfo             ,
#endif
    textNodeGetGlyphs                       ,


-- ** getNumGlyphs #method:getNumGlyphs#

#if defined(ENABLE_OVERLOADING)
    TextNodeGetNumGlyphsMethodInfo          ,
#endif
    textNodeGetNumGlyphs                    ,


-- ** getOffset #method:getOffset#

#if defined(ENABLE_OVERLOADING)
    TextNodeGetOffsetMethodInfo             ,
#endif
    textNodeGetOffset                       ,


-- ** hasColorGlyphs #method:hasColorGlyphs#

#if defined(ENABLE_OVERLOADING)
    TextNodeHasColorGlyphsMethodInfo        ,
#endif
    textNodeHasColorGlyphs                  ,


-- ** new #method:new#

    textNodeNew                             ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Gsk.Callbacks as Gsk.Callbacks
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums
import {-# SOURCE #-} qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Pango.Objects.Font as Pango.Font
import qualified GI.Pango.Structs.GlyphInfo as Pango.GlyphInfo
import qualified GI.Pango.Structs.GlyphString as Pango.GlyphString

#else
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Graphene.Structs.Point as Graphene.Point
import {-# SOURCE #-} qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Pango.Objects.Font as Pango.Font
import qualified GI.Pango.Structs.GlyphInfo as Pango.GlyphInfo
import qualified GI.Pango.Structs.GlyphString as Pango.GlyphString

#endif

-- | Memory-managed wrapper type.
newtype TextNode = TextNode (SP.ManagedPtr TextNode)
    deriving (Eq)

instance SP.ManagedPtrNewtype TextNode where
    toManagedPtr (TextNode p) = p

foreign import ccall "gsk_text_node_get_type"
    c_gsk_text_node_get_type :: IO B.Types.GType

instance B.Types.TypedObject TextNode where
    glibType = c_gsk_text_node_get_type

-- | Type class for types which can be safely cast to t'TextNode', for instance with `toTextNode`.
class (SP.BoxedPtr o, SP.TypedObject o, O.IsDescendantOf TextNode o) => IsTextNode o
instance (SP.BoxedPtr o, SP.TypedObject o, O.IsDescendantOf TextNode o) => IsTextNode o

instance O.HasParentTypes TextNode
type instance O.ParentTypes TextNode = '[Gsk.RenderNode.RenderNode]

-- | Cast to t'TextNode', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toTextNode :: (MIO.MonadIO m, IsTextNode o) => o -> m TextNode
toTextNode = MIO.liftIO . B.ManagedPtr.unsafeCastTo TextNode

--- XXX Missing getter and/or setter, so no GValue instance could be generated.
#if defined(ENABLE_OVERLOADING)
type family ResolveTextNodeMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveTextNodeMethod "draw" o = Gsk.RenderNode.RenderNodeDrawMethodInfo
    ResolveTextNodeMethod "hasColorGlyphs" o = TextNodeHasColorGlyphsMethodInfo
    ResolveTextNodeMethod "ref" o = Gsk.RenderNode.RenderNodeRefMethodInfo
    ResolveTextNodeMethod "serialize" o = Gsk.RenderNode.RenderNodeSerializeMethodInfo
    ResolveTextNodeMethod "unref" o = Gsk.RenderNode.RenderNodeUnrefMethodInfo
    ResolveTextNodeMethod "writeToFile" o = Gsk.RenderNode.RenderNodeWriteToFileMethodInfo
    ResolveTextNodeMethod "getBounds" o = Gsk.RenderNode.RenderNodeGetBoundsMethodInfo
    ResolveTextNodeMethod "getColor" o = TextNodeGetColorMethodInfo
    ResolveTextNodeMethod "getFont" o = TextNodeGetFontMethodInfo
    ResolveTextNodeMethod "getGlyphs" o = TextNodeGetGlyphsMethodInfo
    ResolveTextNodeMethod "getNodeType" o = Gsk.RenderNode.RenderNodeGetNodeTypeMethodInfo
    ResolveTextNodeMethod "getNumGlyphs" o = TextNodeGetNumGlyphsMethodInfo
    ResolveTextNodeMethod "getOffset" o = TextNodeGetOffsetMethodInfo
    ResolveTextNodeMethod "getOpaqueRect" o = Gsk.RenderNode.RenderNodeGetOpaqueRectMethodInfo
    ResolveTextNodeMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveTextNodeMethod t TextNode, O.OverloadedMethod info TextNode p) => OL.IsLabel t (TextNode -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveTextNodeMethod t TextNode, O.OverloadedMethod info TextNode p, R.HasField t TextNode p) => R.HasField t TextNode p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveTextNodeMethod t TextNode, O.OverloadedMethodInfo info TextNode) => OL.IsLabel t (O.MethodProxy info TextNode) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- XXX Wrapping a foreign struct/union with no known destructor or size, leak?
instance BoxedPtr TextNode where
    boxedPtrCopy = return
    boxedPtrFree = \_x -> return ()


-- method TextNode::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "font"
--           , argType = TInterface Name { namespace = "Pango" , name = "Font" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `PangoFont` containing the glyphs"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "glyphs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "GlyphString" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `PangoGlyphString` to render"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "color"
--           , argType = TInterface Name { namespace = "Gdk" , name = "RGBA" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the foreground color to render with"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Point" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "offset of the baseline"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "TextNode" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_text_node_new" gsk_text_node_new :: 
    Ptr Pango.Font.Font ->                  -- font : TInterface (Name {namespace = "Pango", name = "Font"})
    Ptr Pango.GlyphString.GlyphString ->    -- glyphs : TInterface (Name {namespace = "Pango", name = "GlyphString"})
    Ptr Gdk.RGBA.RGBA ->                    -- color : TInterface (Name {namespace = "Gdk", name = "RGBA"})
    Ptr Graphene.Point.Point ->             -- offset : TInterface (Name {namespace = "Graphene", name = "Point"})
    IO (Ptr TextNode)

-- | Creates a render node that renders the given glyphs.
-- 
-- Note that /@color@/ may not be used if the font contains
-- color glyphs.
textNodeNew ::
    (B.CallStack.HasCallStack, MonadIO m, Pango.Font.IsFont a) =>
    a
    -- ^ /@font@/: the @PangoFont@ containing the glyphs
    -> Pango.GlyphString.GlyphString
    -- ^ /@glyphs@/: the @PangoGlyphString@ to render
    -> Gdk.RGBA.RGBA
    -- ^ /@color@/: the foreground color to render with
    -> Graphene.Point.Point
    -- ^ /@offset@/: offset of the baseline
    -> m (Maybe TextNode)
    -- ^ __Returns:__ a new @GskRenderNode@
textNodeNew font glyphs color offset = liftIO $ do
    font' <- unsafeManagedPtrCastPtr font
    glyphs' <- unsafeManagedPtrGetPtr glyphs
    color' <- unsafeManagedPtrGetPtr color
    offset' <- unsafeManagedPtrGetPtr offset
    result <- gsk_text_node_new font' glyphs' color' offset'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapPtr TextNode) result'
        return result''
    touchManagedPtr font
    touchManagedPtr glyphs
    touchManagedPtr color
    touchManagedPtr offset
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method TextNode::get_color
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "TextNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a text `GskRenderNode`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "RGBA" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_text_node_get_color" gsk_text_node_get_color :: 
    Ptr TextNode ->                         -- node : TInterface (Name {namespace = "Gsk", name = "TextNode"})
    IO (Ptr Gdk.RGBA.RGBA)

-- | Retrieves the color used by the text /@node@/.
-- 
-- The value returned by this function will not be correct
-- if the render node was created for a non-sRGB color.
textNodeGetColor ::
    (B.CallStack.HasCallStack, MonadIO m, IsTextNode a) =>
    a
    -- ^ /@node@/: a text @GskRenderNode@
    -> m Gdk.RGBA.RGBA
    -- ^ __Returns:__ the text color
textNodeGetColor node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_text_node_get_color node'
    checkUnexpectedReturnNULL "textNodeGetColor" result
    result' <- (newBoxed Gdk.RGBA.RGBA) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data TextNodeGetColorMethodInfo
instance (signature ~ (m Gdk.RGBA.RGBA), MonadIO m, IsTextNode a) => O.OverloadedMethod TextNodeGetColorMethodInfo a signature where
    overloadedMethod = textNodeGetColor

instance O.OverloadedMethodInfo TextNodeGetColorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.TextNode.textNodeGetColor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-TextNode.html#v:textNodeGetColor"
        })


#endif

-- method TextNode::get_font
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "TextNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The `GskRenderNode`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Font" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_text_node_get_font" gsk_text_node_get_font :: 
    Ptr TextNode ->                         -- node : TInterface (Name {namespace = "Gsk", name = "TextNode"})
    IO (Ptr Pango.Font.Font)

-- | Returns the font used by the text /@node@/.
textNodeGetFont ::
    (B.CallStack.HasCallStack, MonadIO m, IsTextNode a) =>
    a
    -- ^ /@node@/: The @GskRenderNode@
    -> m Pango.Font.Font
    -- ^ __Returns:__ the font
textNodeGetFont node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_text_node_get_font node'
    checkUnexpectedReturnNULL "textNodeGetFont" result
    result' <- (newObject Pango.Font.Font) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data TextNodeGetFontMethodInfo
instance (signature ~ (m Pango.Font.Font), MonadIO m, IsTextNode a) => O.OverloadedMethod TextNodeGetFontMethodInfo a signature where
    overloadedMethod = textNodeGetFont

instance O.OverloadedMethodInfo TextNodeGetFontMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.TextNode.textNodeGetFont",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-TextNode.html#v:textNodeGetFont"
        })


#endif

-- method TextNode::get_glyphs
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "TextNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a text `GskRenderNode`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_glyphs"
--           , argType = TBasicType TUInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of glyphs returned"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_glyphs"
--              , argType = TBasicType TUInt
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of glyphs returned"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just
--               (TCArray
--                  False
--                  (-1)
--                  1
--                  (TInterface Name { namespace = "Pango" , name = "GlyphInfo" }))
-- throws : False
-- Skip return : False

foreign import ccall "gsk_text_node_get_glyphs" gsk_text_node_get_glyphs :: 
    Ptr TextNode ->                         -- node : TInterface (Name {namespace = "Gsk", name = "TextNode"})
    Ptr Word32 ->                           -- n_glyphs : TBasicType TUInt
    IO (Ptr Pango.GlyphInfo.GlyphInfo)

-- | Retrieves the glyph information in the /@node@/.
textNodeGetGlyphs ::
    (B.CallStack.HasCallStack, MonadIO m, IsTextNode a) =>
    a
    -- ^ /@node@/: a text @GskRenderNode@
    -> m [Pango.GlyphInfo.GlyphInfo]
    -- ^ __Returns:__ the glyph information
textNodeGetGlyphs node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    nGlyphs <- allocMem :: IO (Ptr Word32)
    result <- gsk_text_node_get_glyphs node' nGlyphs
    nGlyphs' <- peek nGlyphs
    checkUnexpectedReturnNULL "textNodeGetGlyphs" result
    result' <- (unpackBlockArrayWithLength 24 nGlyphs') result
    result'' <- mapM (newPtr Pango.GlyphInfo.GlyphInfo) result'
    touchManagedPtr node
    freeMem nGlyphs
    return result''

#if defined(ENABLE_OVERLOADING)
data TextNodeGetGlyphsMethodInfo
instance (signature ~ (m [Pango.GlyphInfo.GlyphInfo]), MonadIO m, IsTextNode a) => O.OverloadedMethod TextNodeGetGlyphsMethodInfo a signature where
    overloadedMethod = textNodeGetGlyphs

instance O.OverloadedMethodInfo TextNodeGetGlyphsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.TextNode.textNodeGetGlyphs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-TextNode.html#v:textNodeGetGlyphs"
        })


#endif

-- method TextNode::get_num_glyphs
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "TextNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a text `GskRenderNode`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_text_node_get_num_glyphs" gsk_text_node_get_num_glyphs :: 
    Ptr TextNode ->                         -- node : TInterface (Name {namespace = "Gsk", name = "TextNode"})
    IO Word32

-- | Retrieves the number of glyphs in the text node.
textNodeGetNumGlyphs ::
    (B.CallStack.HasCallStack, MonadIO m, IsTextNode a) =>
    a
    -- ^ /@node@/: a text @GskRenderNode@
    -> m Word32
    -- ^ __Returns:__ the number of glyphs
textNodeGetNumGlyphs node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_text_node_get_num_glyphs node'
    touchManagedPtr node
    return result

#if defined(ENABLE_OVERLOADING)
data TextNodeGetNumGlyphsMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsTextNode a) => O.OverloadedMethod TextNodeGetNumGlyphsMethodInfo a signature where
    overloadedMethod = textNodeGetNumGlyphs

instance O.OverloadedMethodInfo TextNodeGetNumGlyphsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.TextNode.textNodeGetNumGlyphs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-TextNode.html#v:textNodeGetNumGlyphs"
        })


#endif

-- method TextNode::get_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "TextNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a text `GskRenderNode`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Graphene" , name = "Point" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_text_node_get_offset" gsk_text_node_get_offset :: 
    Ptr TextNode ->                         -- node : TInterface (Name {namespace = "Gsk", name = "TextNode"})
    IO (Ptr Graphene.Point.Point)

-- | Retrieves the offset applied to the text.
textNodeGetOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsTextNode a) =>
    a
    -- ^ /@node@/: a text @GskRenderNode@
    -> m Graphene.Point.Point
    -- ^ __Returns:__ a point with the horizontal and vertical offsets
textNodeGetOffset node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_text_node_get_offset node'
    checkUnexpectedReturnNULL "textNodeGetOffset" result
    result' <- (newBoxed Graphene.Point.Point) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data TextNodeGetOffsetMethodInfo
instance (signature ~ (m Graphene.Point.Point), MonadIO m, IsTextNode a) => O.OverloadedMethod TextNodeGetOffsetMethodInfo a signature where
    overloadedMethod = textNodeGetOffset

instance O.OverloadedMethodInfo TextNodeGetOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.TextNode.textNodeGetOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-TextNode.html#v:textNodeGetOffset"
        })


#endif

-- method TextNode::has_color_glyphs
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "TextNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a text `GskRenderNode`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_text_node_has_color_glyphs" gsk_text_node_has_color_glyphs :: 
    Ptr TextNode ->                         -- node : TInterface (Name {namespace = "Gsk", name = "TextNode"})
    IO CInt

-- | Checks whether the text /@node@/ has color glyphs.
-- 
-- /Since: 4.2/
textNodeHasColorGlyphs ::
    (B.CallStack.HasCallStack, MonadIO m, IsTextNode a) =>
    a
    -- ^ /@node@/: a text @GskRenderNode@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the text node has color glyphs
textNodeHasColorGlyphs node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_text_node_has_color_glyphs node'
    let result' = (/= 0) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data TextNodeHasColorGlyphsMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsTextNode a) => O.OverloadedMethod TextNodeHasColorGlyphsMethodInfo a signature where
    overloadedMethod = textNodeHasColorGlyphs

instance O.OverloadedMethodInfo TextNodeHasColorGlyphsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.TextNode.textNodeHasColorGlyphs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-TextNode.html#v:textNodeHasColorGlyphs"
        })


#endif


