/*
 * mousegrabber.c - mouse pointer grabbing
 *
 * Copyright © 2008-2009 Julien Danjou <julien@danjou.info>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/** awesome mousegrabber API
 * @author Julien Danjou &lt;julien@danjou.info&gt;
 * @copyright 2008-2009 Julien Danjou
 * @module mousegrabber
 */

#include "mousegrabber.h"
#include "common/xcursor.h"
#include "mouse.h"
#include "globalconf.h"

#include <unistd.h>
#include <stdbool.h>

/** Grab the mouse.
 * \param cursor The cursor to use while grabbing.
 * \return True if mouse was grabbed.
 */
static bool
mousegrabber_grab(xcb_cursor_t cursor)
{
    xcb_window_t root = globalconf.screen->root;

    for(int i = 1000; i; i--)
    {
        xcb_grab_pointer_reply_t *grab_ptr_r;
        xcb_grab_pointer_cookie_t grab_ptr_c =
            xcb_grab_pointer_unchecked(globalconf.connection, false, root,
                                       XCB_EVENT_MASK_BUTTON_PRESS
                                       | XCB_EVENT_MASK_BUTTON_RELEASE
                                       | XCB_EVENT_MASK_POINTER_MOTION,
                                       XCB_GRAB_MODE_ASYNC,
                                       XCB_GRAB_MODE_ASYNC,
                                       root, cursor, XCB_CURRENT_TIME);

        if((grab_ptr_r = xcb_grab_pointer_reply(globalconf.connection, grab_ptr_c, NULL)))
        {
            p_delete(&grab_ptr_r);
            return true;
        }
        usleep(1000);
    }
    return false;
}

/** Handle mouse motion events.
 * \param L Lua stack to push the pointer motion.
 * \param x The received mouse event x component.
 * \param y The received mouse event y component.
 * \param mask The received mouse event bit mask.
 */
void
mousegrabber_handleevent(lua_State *L, int x, int y, uint16_t mask)
{
    luaA_mouse_pushstatus(L, x, y, mask);
}

/** Grab the mouse pointer and list motions, calling callback function at each
 * motion. The callback function must return a boolean value: true to
 * continue grabbing, false to stop.
 * The function is called with one argument:
 * a table containing modifiers pointer coordinates.
 *
 * The list of valid cusrors are:
 *
 * * <table class='widget_list' border=1>
 * <tr><td>num_glyphs          </td></tr>
 * <tr><td>cursor              </td></tr>
 * <tr><td>arrow               </td></tr>
 * <tr><td>based_arrow_down    </td></tr>
 * <tr><td>based_arrow_up      </td></tr>
 * <tr><td>boat                </td></tr>
 * <tr><td>bogosity            </td></tr>
 * <tr><td>bottom_left_corner  </td></tr>
 * <tr><td>bottom_right_corner </td></tr>
 * <tr><td>bottom_side         </td></tr>
 * <tr><td>bottom_tee          </td></tr>
 * <tr><td>box_spiral          </td></tr>
 * <tr><td>center_ptr          </td></tr>
 * <tr><td>circle              </td></tr>
 * <tr><td>clock               </td></tr>
 * <tr><td>coffee_mug          </td></tr>
 * <tr><td>cross               </td></tr>
 * <tr><td>cross_reverse       </td></tr>
 * <tr><td>crosshair           </td></tr>
 * <tr><td>diamond_cross       </td></tr>
 * <tr><td>dot                 </td></tr>
 * <tr><td>dotbox              </td></tr>
 * <tr><td>double_arrow        </td></tr>
 * <tr><td>draft_large         </td></tr>
 * <tr><td>draft_small         </td></tr>
 * <tr><td>draped_box          </td></tr>
 * <tr><td>exchange            </td></tr>
 * <tr><td>fleur               </td></tr>
 * <tr><td>gobbler             </td></tr>
 * <tr><td>gumby               </td></tr>
 * <tr><td>hand                </td></tr>
 * <tr><td>hand                </td></tr>
 * <tr><td>heart               </td></tr>
 * <tr><td>icon                </td></tr>
 * <tr><td>iron_cross          </td></tr>
 * <tr><td>left_ptr            </td></tr>
 * <tr><td>left_side           </td></tr>
 * <tr><td>left_tee            </td></tr>
 * <tr><td>leftbutton          </td></tr>
 * <tr><td>ll_angle            </td></tr>
 * <tr><td>lr_angle            </td></tr>
 * <tr><td>man                 </td></tr>
 * <tr><td>middlebutton        </td></tr>
 * <tr><td>mouse               </td></tr>
 * <tr><td>pencil              </td></tr>
 * <tr><td>pirate              </td></tr>
 * <tr><td>plus                </td></tr>
 * <tr><td>question_arrow      </td></tr>
 * <tr><td>right_ptr           </td></tr>
 * <tr><td>right_side          </td></tr>
 * <tr><td>right_tee           </td></tr>
 * <tr><td>rightbutton         </td></tr>
 * <tr><td>rtl_logo            </td></tr>
 * <tr><td>sailboat            </td></tr>
 * <tr><td>sb_down_arrow       </td></tr>
 * <tr><td>sb_h_double_arrow   </td></tr>
 * <tr><td>sb_left_arrow       </td></tr>
 * <tr><td>sb_right_arrow      </td></tr>
 * <tr><td>sb_up_arrow         </td></tr>
 * <tr><td>sb_v_double_arrow   </td></tr>
 * <tr><td>shuttle             </td></tr>
 * <tr><td>sizing              </td></tr>
 * <tr><td>spider              </td></tr>
 * <tr><td>spraycan            </td></tr>
 * <tr><td>star                </td></tr>
 * <tr><td>target              </td></tr>
 * <tr><td>tcross              </td></tr>
 * <tr><td>top_left_arrow      </td></tr>
 * <tr><td>top_left_corner     </td></tr>
 * <tr><td>top_right_corner    </td></tr>
 * <tr><td>top_side            </td></tr>
 * <tr><td>top_tee             </td></tr>
 * <tr><td>trek                </td></tr>
 * <tr><td>ul_angle            </td></tr>
 * <tr><td>umbrella            </td></tr>
 * <tr><td>ur_angle            </td></tr>
 * <tr><td>watch               </td></tr>
 * <tr><td>xterm               </td></tr>
 * </table>
 *
 *
 * @param func A callback function as described above.
 * @param cursor The name of a X cursor to use while grabbing.
 * @function run
 */
static int
luaA_mousegrabber_run(lua_State *L)
{
    if(globalconf.mousegrabber != LUA_REFNIL)
        luaL_error(L, "mousegrabber already running");

    uint16_t cfont = xcursor_font_fromstr(luaL_checkstring(L, 2));

    if(cfont)
    {
        xcb_cursor_t cursor = xcursor_new(globalconf.cursor_ctx, cfont);

        luaA_registerfct(L, 1, &globalconf.mousegrabber);

        if(!mousegrabber_grab(cursor))
        {
            luaA_unregister(L, &globalconf.mousegrabber);
            luaL_error(L, "unable to grab mouse pointer");
        }
    }
    else
        luaA_warn(L, "invalid cursor");

    return 0;
}

/** Stop grabbing the mouse pointer.
 *
 * @function stop
 */
int
luaA_mousegrabber_stop(lua_State *L)
{
    xcb_ungrab_pointer(globalconf.connection, XCB_CURRENT_TIME);
    luaA_unregister(L, &globalconf.mousegrabber);
    return 0;
}

/** Check if mousegrabber is running.
 *
 * @treturn boolean True if running, false otherwise.
 * @function isrunning
 */
static int
luaA_mousegrabber_isrunning(lua_State *L)
{
    lua_pushboolean(L, globalconf.mousegrabber != LUA_REFNIL);
    return 1;
}

const struct luaL_Reg awesome_mousegrabber_lib[] =
{
    { "run", luaA_mousegrabber_run },
    { "stop", luaA_mousegrabber_stop },
    { "isrunning", luaA_mousegrabber_isrunning },
    { "__index", luaA_default_index },
    { "__newindex", luaA_default_newindex },
    { NULL, NULL }
};

// vim: filetype=c:expandtab:shiftwidth=4:tabstop=8:softtabstop=4:textwidth=80
